<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
class ProfileController extends Controller
{
public function updateWalletAddress(Request $request)
{
    $request->validate([
        'withdraw_account' => 'required|string|max:255',
        'gateway_method' => 'required|string|max:50',
    ]);

    try {
        $user = Auth::user();

        if (!$user) {
            return back()->with('error', 'User not authenticated.');
        }

        $user->withdraw_account = $request->withdraw_account;
        $user->gateway_method = $request->gateway_method; // Ensure this column exists in DB

        $user->save();

        return back()->with('success', 'Wallet address updated successfully!');
    } catch (\Exception $e) {
        Log::error('Wallet update error: ' . $e->getMessage(), [
            'user_id' => Auth::id(),
            'request_data' => $request->all()
        ]);
        return back()->with('error', 'Something went wrong! Please try again later.');
    }
}
public function update(Request $request)
{
    $user = Auth::user();

    // Validation (making all fields optional)
    $request->validate([
        'email' => 'nullable|email',
        'withdraw_account' => 'nullable|string',
        'profile_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        'password' => 'nullable|min:6',
    ]);

    try {
        // Update email if provided
        if ($request->filled('email')) {
            $user->email = $request->email;
        }

        // Update wallet address if provided
        if ($request->filled('withdraw_account')) {
            $user->withdraw_account = $request->withdraw_account;
        }

        // Check if a profile image has been uploaded and handle the upload
        if ($request->hasFile('profile_image')) {
            $image = $request->file('profile_image');
            $imageName = time() . '_' . $image->getClientOriginalName();
            $image->move(public_path('images'), $imageName);
            $user->profile_image = $imageName;
        }

        // Update password if provided
        if ($request->filled('password')) {
            $user->password = bcrypt($request->password);
        }

        // Save user changes
        $user->save();

        return redirect()->back()->with('success', 'Profile updated successfully.');
    } catch (\Exception $e) {
        // Log the error and return an error message
        \Log::error('Profile update failed: ' . $e->getMessage());
        return redirect()->back()->with('error', 'Failed to update profile. Please try again.');
    }
}
    // Show the reset password form
    public function showResetPasswordForm()
    {
        return view('reset-password'); // Make sure this matches the name of your view file
    }

    // Handle the password reset
    public function resetPassword(Request $request)
    {
        // Validate the input
        $validator = Validator::make($request->all(), [
            'old_password' => 'required',
            'new_password' => 'required|confirmed|min:8',
        ]);

        // Check if the validation failed
        if ($validator->fails()) {
            return redirect()->route('reset-password')
                ->withErrors($validator)
                ->withInput();
        }

        $user = Auth::user();

        // Check if the old password matches
        if (Hash::check($request->old_password, $user->password)) {
            // Update the password
            $user->password = Hash::make($request->new_password);
            $user->save();

            return redirect()->route('reset-password')->with('success', 'Password reset successfully.');
        } else {
            return redirect()->route('reset-password')->with('error', 'Old password is incorrect.');
        }
    }
   public function edit()
    {
        return view('settings.settings');
    }
}