<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class QuestionController extends Controller
{
    public function showQuestions()
    {
        $user = Auth::user();
        $vip = $user->vip_level ?? 1;
        $vipToQuestionCount = [1 => 2, 2 => 4, 3 => 6, 4 => 8, 5 => 10, 6 => 12];
        $questionCount = $vipToQuestionCount[$vip] ?? 2;

        $today = Carbon::today();
        $questionsByPackage = [];

        $investments = DB::table('invest')
            ->where('user_id', $user->id)
            ->where('status', 1)
            ->get()
            ->filter(function ($inv) {
                return Carbon::parse($inv->date)->diffInDays(now()) < 60;
            });

        if ($investments->isEmpty()) {
            return redirect('/none')->with('info', 'You have no active investments.');
        }

        foreach ($investments as $investment) {
            $alreadyAnswered = DB::table('user_daily_questions')
                ->where('user_id', $user->id)
                ->where('package_id', $investment->package_id)
                ->where('capital', $investment->capital)
                ->whereDate('question_date', $today)
                ->exists();

            if ($alreadyAnswered) {
                continue;
            }

            $cacheKey = "quiz_{$user->id}_pkg{$investment->package_id}_cap{$investment->capital}_{$today->toDateString()}";
            $questions = cache()->remember($cacheKey, 86400, function () use ($questionCount) {
                return DB::table('quiz')->inRandomOrder()->limit($questionCount)->get();
            });

            $questionsByPackage[$investment->id] = [
                'package_id' => $investment->package_id,
                'capital' => $investment->capital,
                'questions' => $questions,
            ];
        }

        if (empty($questionsByPackage)) {
            return redirect('/answered')->with('info', 'You’ve completed all tasks for today. Come back tomorrow!');
        }

        return view('daily.questions', compact('questionsByPackage'));
    }

    public function submitAnswers(Request $request)
    {
        $user = Auth::user();
        $today = Carbon::today()->toDateString();

        $validated = $request->validate([
            'answers' => 'required|array',
            'answers.*.question_id' => 'required|integer',
            'answers.*.selected_answer' => 'required|string',
            'answers.*.package_id' => 'required|integer',
            'answers.*.capital' => 'required|numeric',
        ]);

        $answers = $validated['answers'];

        $rewardedPackages = collect($answers)->unique(function ($a) {
            return $a['package_id'] . '_' . $a['capital'];
        });

        foreach ($answers as $entry) {
            $questionId = $entry['question_id'];
            $selectedAnswer = $entry['selected_answer'];
            $packageId = $entry['package_id'];
            $capital = $entry['capital'];

            $alreadyAnswered = DB::table('user_daily_questions')
                ->where('user_id', $user->id)
                ->where('question_id', $questionId)
                ->whereDate('question_date', $today)
                ->exists();

            if ($alreadyAnswered) {
                continue;
            }

            $correctAnswer = DB::table('quiz')->where('id', $questionId)->value('correct_answer');
            $isCorrect = $selectedAnswer === $correctAnswer;

            DB::table('user_daily_questions')->insert([
                'user_id' => $user->id,
                'question_id' => $questionId,
                'package_id' => $packageId,
                'capital' => $capital,
                'question_date' => $today,
                'selected_answer' => $selectedAnswer,
                'is_correct' => $isCorrect,
                'created_at' => now(),
                'updated_at' => now(),
            ]);
        }

        $totalReward = 0;

        foreach ($rewardedPackages as $pkg) {
            $packageId = $pkg['package_id'];
            $capital = $pkg['capital'];

            $percentage = DB::table('packages')->where('id', $packageId)->value('daily_income');
            if ($percentage === null) continue;

            $reward = $capital * $percentage / 100;
            $totalReward += $reward;

            // Credit user balance
            DB::table('users')->where('id', $user->id)->increment('balance', $reward);

            // Increment days_paid
            DB::table('invest')
                ->where('user_id', $user->id)
                ->where('package_id', $packageId)
                ->where('capital', $capital)
                ->where('status', 1)
                ->increment('days_paid');

            // Check maturity and update status
            $investment = DB::table('invest')
                ->where('user_id', $user->id)
                ->where('package_id', $packageId)
                ->where('capital', $capital)
                ->first();

            if ($investment && $investment->days_paid >= $investment->period) {
                DB::table('invest')
                    ->where('id', $investment->id)
                    ->update(['status' => 2]);
            }
        }

        return redirect()->route('dashboard')->with('success', "You earned {$totalReward} for completing today's tasks.");
    }
}
