<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class VideoTaskController extends Controller
{
    private $freeReward = 20; // Ksh 20 reward for free task
    private $paidReward = 100; // Default Ksh reward for paid task (or calculate per investment)

    public function videoTask()
    {
        $user = Auth::user();
        $invests = DB::table('invest')->where('user_id', $user->id)->get();

        // Free task cooldown
        $lastFreeTask = DB::table('video_tasks')
            ->where('user_id', $user->id)
            ->where('type', 'free')
            ->latest('created_at')
            ->first();

        $freeTaskStatus = [
            'available' => true,
            'cooldown' => null
        ];

        if ($lastFreeTask) {
            $next = Carbon::parse($lastFreeTask->created_at)->addHours(24);
            if (now()->lt($next)) {
                $freeTaskStatus['available'] = false;
                $freeTaskStatus['cooldown'] = $next;
            }
        }

        // Sample video (static or fetched dynamically)
        $video = [
            'videoId' => 'dQw4w9WgXcQ',
            'title' => 'Daily Earnings Tutorial',
            'channel' => 'YourChannel',
            'duration' => '00:45'
        ];

        return view('video-task', [
            'freeTaskStatus' => $freeTaskStatus,
            'video' => $video,
            'invests' => $invests
        ]);
    }

    public function store(Request $request)
{
    Log::info($request->all());
   $validated = $request->validate([
        'type' => 'required|in:free,paid',
       'investment_id' => 'nullable|integer|exists:invest,id', // if applicable
        'video_id' => 'sometimes|string' // if you need it
    ]);

  Log::info('hy after $validated');
    $user = Auth::user();
    $type = $validated['type'];
    $investmentId = $validated['investment_id'] ?? null;

    try {
        if ($type === 'free') {
            Log::info('hy in if statem free');
            return $this->handleFreeTask($user);
        }

        return $this->handlePaidTask($user, $investmentId);
    } catch (\Exception $e) {
        Log::error("Video task failed: " . $e->getMessage(), [
            'user_id' => $user->id,
            'type' => $type,
            'investment_id' => $investmentId
        ]);
        
        return response()->json([
            'success' => false,
            'message' => 'An error occurred while processing your task.'
        ], 500);
    }
}

protected function handleFreeTask($user)
{
    // Check cooldown using a query scope
    $lastFreeTask = DB::table('video_tasks')
        ->where('user_id', $user->id)
        ->where('type', 'free')
        ->latest('created_at')
        ->first();

    if ($lastFreeTask && Carbon::parse($lastFreeTask->created_at)->addHours(24)->isFuture()) {
        return response()->json([
            'success' => false,
            'message' => 'Free task already completed. Try again later.'
        ], 429); // 429 Too Many Requests
    }

DB::transaction(function () use ($user) {
    // Increment both balance and video_rewards
    DB::table('users')
        ->where('id', $user->id)
        ->update([
            'balance' => DB::raw('balance + ' . $this->freeReward),
            'video_rewards' => DB::raw('video_rewards + ' . $this->freeReward),
        ]);

        // Log task
        DB::table('video_tasks')->insert([
            'user_id' => $user->id,
            'type' => 'free',
            'amount' => $this->freeReward,
            'created_at' => now(),
            'updated_at' => now()
        ]);
    });

    return response()->json([
        'success' => true,
        'message' => '🎉 Free task completed! You earned Ksh ' . $this->freeReward,
        'amount' => $this->freeReward
    ]);
}

protected function handlePaidTask($user, $investmentId)
{
    $invest = DB::table('invest')
        ->where('id', $investmentId)
        ->where('user_id', $user->id)
        ->first();

    if (!$invest) {
        return response()->json([
            'success' => false,
            'message' => 'Investment not found.'
        ], 404);
    }

    // Check cooldown using a query scope
    $lastPaidTask = DB::table('video_tasks')
        ->where('user_id', $user->id)
        ->where('type', 'paid')
        ->where('invest_id', $investmentId)
        ->latest('created_at')
        ->first();

    if ($lastPaidTask && Carbon::parse($lastPaidTask->created_at)->addHours(24)->isFuture()) {
        return response()->json([
            'success' => false,
            'message' => 'You already completed today\'s task for this investment.'
        ], 429);
    }

    // Calculate reward safely
    $reward = $this->calculateDailyReward($invest->returns, $invest->daily_tasks);

    DB::transaction(function () use ($user, $investmentId, $reward) {
    // Increment both balance and video_rewards
    DB::table('users')
        ->where('id', $user->id)
        ->update([
            'balance' => DB::raw('balance + ' . $reward),
            'video_rewards' => DB::raw('video_rewards + ' . $reward),
        ]);
        DB::table('invest')
            ->where('id', $investmentId)
            ->increment('days_paid');

        // Log task
        DB::table('video_tasks')->insert([
            'user_id' => $user->id,
            'invest_id' => $investmentId,
            'type' => 'paid',
            'amount' => $reward,
            'created_at' => now(),
            'updated_at' => now()
        ]);
    });

    return response()->json([
        'success' => true,
        'message' => "✅ Paid task completed! Ksh $reward has been added.",
        'amount' => $reward
    ]);
}

protected function calculateDailyReward($totalReturns, $dailyTasks)
{
    if ($dailyTasks <= 0) return 0;
    return floor($totalReturns / max($dailyTasks, 1)); // Prevent division by zero
}
    // public function store(Request $request)
    // {
    //     Log::info($request);
    //     $user = Auth::user();
    //     $type = $request->type;
    //     $investId = $request->invest_id;

    //     if ($type === 'free') {
    //         // Check cooldown
    //         $lastFreeTask = DB::table('video_tasks')
    //             ->where('user_id', $user->id)
    //             ->where('type', 'free')
    //             ->latest('created_at')
    //             ->first();

    //         if ($lastFreeTask && Carbon::parse($lastFreeTask->created_at)->addHours(24)->isFuture()) {
    //             return response()->json([
    //                 'success' => false,
    //                 'message' => 'Free task already completed. Try again later.'
    //             ]);
    //         }

    //         // Reward logic
    //         DB::table('users')->where('id', $user->id)->increment('balance', $this->freeReward);

    //         // Log task
    //         DB::table('video_tasks')->insert([
    //             'user_id' => $user->id,
    //             'type' => 'free',
    //             'amount' => $this->freeReward,
    //             'created_at' => now(),
    //             'updated_at' => now()
    //         ]);

    //         return response()->json([
    //             'success' => true,
    //             'message' => 'ðŸŽ‰ Free task completed! You earned Ksh ' . $this->freeReward
    //         ]);
    //     }

    //     if ($type === 'paid') {
    //         if (!$investId) {
    //             Log::info('Hey');
    //             return response()->json(['success' => false, 'message' => 'Invalid investment.']);
    //         }

    //         $invest = DB::table('invest')
    //             ->where('id', $investId)
    //             ->where('user_id', $user->id)
    //             ->first();
            
    //         if (!$invest) {
    //             return response()->json(['success' => false, 'message' => 'Investment not found.']);
    //         }

    //         // Cooldown
    //         $lastPaidTask = DB::table('video_tasks')
    //             ->where('user_id', $user->id)
    //             ->where('type', 'paid')
    //             ->where('invest_id', $investId)
    //             ->latest('created_at')
    //             ->first();

    //         if ($lastPaidTask && Carbon::parse($lastPaidTask->created_at)->addHours(24)->isFuture()) {
    //             return response()->json([
    //                 'success' => false,
    //                 'message' => 'You already did today\'s task for this investment.'
    //             ]);
    //         }

    //         // Reward logic (custom per invest or static)
    //         $reward = floor($invest->returns / $invest->daily_tasks);

    //         // Update balance and investment days_paid
    //         DB::table('users')->where('id', $user->id)->increment('balance', $reward);
    //         DB::table('invest')->where('id', $investId)->increment('days_paid');

    //         // Log task
    //         DB::table('video_tasks')->insert([
    //             'user_id' => $user->id,
    //             'invest_id' => $investId,
    //             'type' => 'paid',
    //             'amount' => $reward,
    //             'created_at' => now(),
    //             'updated_at' => now()
    //         ]);

    //         return response()->json([
    //             'success' => true,
    //             'message' => "âœ… Paid task completed! Ksh $reward has been added."
    //         ]);
    //     }

    //     return response()->json(['success' => false, 'message' => 'Unknown task type.']);
    // }
}
