@extends('layouts.main')

@section('content')
<?php
    use Carbon\Carbon;
    
    // Video rotation system - changes every 12 hours
    $videoRotationInterval = 12 * 60 * 60; // 12 hours in seconds
    $currentTime = now()->timestamp;
    $videoIndex = floor($currentTime / $videoRotationInterval);
    
    // Your video pool (replace with your actual videos)
    $videoPool = [
        [
            'videoId' => 'dQw4w9WgXcQ',
            'title' => 'Video One',
            'channel' => 'Channel One',
            'duration' => '03:45'
        ],
        [
            'videoId' => '9bZkp7q19f0',
            'title' => 'Video Two', 
            'channel' => 'Channel Two',
            'duration' => '04:20'
        ],
        [
            'videoId' => 'JGwWNGJdvx8',
            'title' => 'Video Three',
            'channel' => 'Channel Three',
            'duration' => '03:30'
        ]
    ];
    
    // Select current video
    $selectedVideoIndex = $videoIndex % count($videoPool);
    $video = $videoPool[$selectedVideoIndex];

    // Authentication and user data
    $user = Auth::user();
    $hasInvestment = $user->investments > 0;
    $freeReward = 20;

    // Free task cooldown logic
    $lastFreeTask = DB::table('video_tasks')
        ->where('user_id', $user->id)
        ->where('type', 'free')
        ->latest('created_at')
        ->first();

    $freeCooldownActive = false;
    $freeCooldownEnd = null;

    if ($lastFreeTask) {
        $cooldownEnd = Carbon::parse($lastFreeTask->created_at)->addHours(24);
        $freeCooldownActive = now()->lt($cooldownEnd);
        $freeCooldownEnd = $freeCooldownActive ? $cooldownEnd : null;
    }

    // Paid investments data
    $investments = $hasInvestment 
        ? DB::table('invest')->where('user_id', $user->id)->get()
        : collect();
?>

<style>
    .video-task-container {
        max-width: 400px;
        margin: 0 auto;
        background-color: black;
        min-height: 100vh;
        padding: 15px;
    }
    .card {
     background: linear-gradient(to left, purple 0%,  #000 100%);
     color:#fff;
        padding:10px;
        border: 1px solid white;
        box-shadow: 0 0 8px #00ff80;
        text-decoration: underline;
        text-decoration-color: black;
    }
    .video-header {
        margin-bottom: 15px;
    }
    .video-meta {
        color: #666;
        font-size: 0.9rem;
    }
    .video-player-container {
        position: relative;
        padding-bottom: 56.25%;
        height: 0;
        overflow: hidden;
        margin-bottom: 15px;
    }
    .video-frame {
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        border-radius: 8px;
    }
    .task-card {
        border-radius: 8px;
        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        padding: 15px;
        margin-bottom: 15px;
    }
    .btn {
        display: block;
        width: 100%;
        padding: 10px;
        margin-bottom: 10px;
        font-weight: bold;
        border: none;
        border-radius: 5px;
        text-align: center;
        color: white;
        cursor: pointer;
        transition: opacity 0.3s;
    }
    .btn-primary {
        background: linear-gradient(to right, #4CAF50, #2E7D32);
    }
    .btn-secondary {
        background: #9E9E9E;
    }
    .btn-success {
        background: linear-gradient(to right, #9C27B0, #6A1B9A);
    }
    .btn-disabled {
        opacity: 0.6;
        pointer-events: none;
    }
    .status-message {
        color: #2E7D32;
        font-weight: bold;
        margin: 10px 0;
        min-height: 20px;
    }
    .cooldown-button {
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 5px;
    }
    .unmute-btn {
        position: absolute;
        bottom: 10px;
        right: 10px;
        z-index: 10;
        background: rgba(0,0,0,0.7);
        color: white;
        border: none;
        padding: 5px 10px;
        border-radius: 4px;
        cursor: pointer;
    }
    .video-rotation-info {
        color: #00ff80;
        font-size: 12px;
        margin-bottom: 10px;
        text-align: center;
    }
</style>

<div class="video-task-container">
    <div class="video-rotation-info">
        Video changes every 12 hours | Next: <?php 
            $nextRotation = ($videoIndex + 1) * $videoRotationInterval;
            echo Carbon::createFromTimestamp($nextRotation)->diffForHumans();
        ?>
    </div>
    
    <div class="video-header">
        <h4>🎬 {{ $video['title'] }}</h4>
        <p class="video-meta">📺 {{ $video['channel'] }} | ⏱️ Duration: {{ $video['duration'] }}</p>
    </div>

    @if($hasInvestment)
        @foreach($investments as $investment)
            @php
                $lastPaidTask = DB::table('video_tasks')
                    ->where('user_id', $user->id)
                    ->where('type', 'paid')
                    ->where('invest_id', $investment->id)
                    ->latest('created_at')
                    ->first();

                $paidCooldownActive = false;
                $paidCooldownEnd = null;

                if ($lastPaidTask) {
                    $cooldownEnd = Carbon::parse($lastPaidTask->created_at)->addHours(24);
                    $paidCooldownActive = now()->lt($cooldownEnd);
                    $paidCooldownEnd = $paidCooldownActive ? $cooldownEnd : null;
                }
            @endphp

            <div class="task-card" data-investment-id="{{ $investment->id }}">
                <span class="card">
                    <h5>📦 Package: {{ $investment->package_id }}</h5>
                    <p>💰 Capital: Ksh {{ number_format($investment->capital, 2) }}</p>
                    <p>📈 Expected Returns: Ksh {{ number_format($investment->returns, 2) }}</p>
                </span>

                <div class="video-player-container" style="border: 1px solid white;">
                    <iframe id="paidVideo{{ $investment->id }}"
                            class="video-frame"
                            src="https://www.youtube.com/embed/{{ $video['videoId'] }}?enablejsapi=1&autoplay=0&mute=1"
                            allowfullscreen
                            allow="autoplay">
                    </iframe>
                    <button class="unmute-btn" data-video-id="paidVideo{{ $investment->id }}">🔇 Unmute</button>
                </div>

                <div id="statusMsg{{ $investment->id }}" class="status-message"></div>

                @if($paidCooldownActive)
                    <button class="btn btn-secondary btn-disabled cooldown-button" style="font-size:12px;">
                        ⏳ Paid task available in 
                        <span id="paidCooldownTimer{{ $investment->id }}" 
                              data-cooldown-end="{{ $paidCooldownEnd->toIso8601String() }}">
                            {{ $paidCooldownEnd->diffForHumans(now(), ['parts' => 2]) }}
                        </span>
                    </button>
                @else
                    <button class="btn btn-success task-button"
                            data-task-type="paid"
                            data-investment-id="{{ $investment->id }}"
                            data-reward="{{ $investment->returns }}"
                            data-video-element="paidVideo{{ $investment->id }}"
                            data-status-element="statusMsg{{ $investment->id }}"
                            data-current-video-id="{{ $video['videoId'] }}">
                        🎯 Claim Paid Task (Earn: {{ $investment->returns }})
                    </button>
                @endif
            </div>
        @endforeach
       <hr><br><br><br><br>
    @else
        <div class="video-player-container" style="border: 1px solid white;">
            <iframe id="videoFrame"
                    class="video-frame"
                    src="https://www.youtube.com/embed/{{ $video['videoId'] }}?enablejsapi=1&autoplay=0&mute=1"
                    allowfullscreen
                    allow="autoplay">
            </iframe>
            <button class="unmute-btn" data-video-id="videoFrame">🔇 Unmute</button>
        </div>

        <div id="statusMsg" class="status-message"></div>

        @if($freeCooldownActive)
            <button class="btn btn-secondary btn-disabled cooldown-button">
                ⏳ Free task available in 
                <span id="freeCooldownTimer" 
                      data-cooldown-end="{{ $freeCooldownEnd->toIso8601String() }}">
                    {{ $freeCooldownEnd->diffForHumans(now(), ['parts' => 2]) }}
                </span>
            </button>
        @else
            <button class="btn btn-primary task-button"
                    data-task-type="free"
                    data-reward="{{ $freeReward }}"
                    data-video-element="videoFrame"
                    data-status-element="statusMsg"
                    data-current-video-id="{{ $video['videoId'] }}">
                🎁 Claim Free Task (Ksh {{ $freeReward }})
            </button>
        @endif
    @endif
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Initialize all cooldown timers
    document.querySelectorAll('[id$="CooldownTimer"]').forEach(timerElement => {
        const endTime = new Date(timerElement.dataset.cooldownEnd).getTime();
        
        const updateTimer = () => {
            const now = new Date().getTime();
            const distance = endTime - now;
            
            if (distance < 0) {
                location.reload();
                return;
            }
            
            const hours = Math.floor((distance % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
            const minutes = Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60));
            const seconds = Math.floor((distance % (1000 * 60)) / 1000);
            
            timerElement.textContent = `${hours}h ${minutes}m ${seconds}s`;
        };
        
        updateTimer();
        setInterval(updateTimer, 1000);
    });

    // Unmute button functionality with medium volume
    document.querySelectorAll('.unmute-btn').forEach(button => {
        button.addEventListener('click', function() {
            const videoId = this.dataset.videoId;
            const iframe = document.getElementById(videoId);
            
            // Unmute and set volume to 50%
            iframe.contentWindow.postMessage(JSON.stringify({
                event: 'command',
                func: 'unMute',
                args: []
            }), '*');
            
            setTimeout(() => {
                iframe.contentWindow.postMessage(JSON.stringify({
                    event: 'command',
                    func: 'setVolume',
                    args: [50]
                }), '*');
            }, 100);
            
            this.textContent = '🔊 50%';
            this.style.pointerEvents = 'none';
        });
    });

    // Task button handler with video rotation
    document.querySelectorAll('.task-button').forEach(button => {
        button.addEventListener('click', function() {
            const type = this.dataset.taskType;
            const investmentId = this.dataset.investmentId || null;
            const videoElement = document.getElementById(this.dataset.videoElement);
            const statusElement = document.getElementById(this.dataset.statusElement);
            const currentVideoId = this.dataset.currentVideoId;
            
            this.classList.add('btn-disabled');
            this.textContent = '⏳ Watching...';
            
            videoElement?.contentWindow?.postMessage(JSON.stringify({
                event: 'command',
                func: 'playVideo',
                args: []
            }), '*');
            
            setTimeout(() => {
                fetch('{{ route("video-task.store") }}', {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}',
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({ 
                        type, 
                        investment_id: investmentId,
                        video_id: currentVideoId
                    })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        this.textContent = '✅ Task Completed';
                        if (statusElement) {
                            statusElement.textContent = data.message;
                        }
                        // Force video rotation after completion
                        setTimeout(() => location.reload(), 2000);
                    } else {
                        this.textContent = '⚠️ Task Failed';
                        if (statusElement) {
                            statusElement.textContent = data.message;
                        }
                        this.classList.remove('btn-disabled');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    this.textContent = '⚠️ Error Occurred';
                    this.classList.remove('btn-disabled');
                });
            }, 10000);
        });
    });

    // Check for video rotation every 5 minutes
    setInterval(() => {
        fetch(window.location.href, {
            headers: {
                'X-Requested-With': 'XMLHttpRequest',
                'Accept': 'application/json'
            }
        })
        .then(response => response.text())
        .then(html => {
            // Simple check if video should rotate
            const parser = new DOMParser();
            const doc = parser.parseFromString(html, 'text/html');
            const rotationInfo = doc.querySelector('.video-rotation-info')?.textContent;
            if (rotationInfo && !rotationInfo.includes(document.querySelector('.video-rotation-info').textContent)) {
                location.reload();
            }
        });
    }, 300000); // 5 minutes
});
</script>

@endsection